% 3D Time Reversal Reconstruction For A Spherical Sensor Example
%
% This example demonstrates the use of k-Wave for the time-reversal
% reconstruction of a three-dimensional photoacoustic wave-field recorded
% over a spherical sensor. The sensor data is simulated and then
% time-reversed using kspaceFirstOrder3D. It builds on the 3D Time Reversal
% Reconstruction For A Planar Sensor Example. 
%
% author: Bradley Treeby
% date: 15th July 2009
% last update: 19th January 2010
%  
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010, 2011 Bradley Treeby and Ben Cox

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>. 

clear all;

% =========================================================================
% SIMULATION
% =========================================================================

% create the computational grid
PML_size = 10;          % size of the PML in pixels
Nx = 64 - 2*PML_size;   % number of pixels in the x direction
Ny = Nx;                % number of pixels in the y direction
Nz = Nx;                % number of pixels in the z direction
dx = 0.2e-3;            % pixel width [m]
dy = dx;                % pixel length [m]
dz = dx;                % pixel height [m]
kgrid = makeGrid(Nx, dx, Ny, dy, Nz, dz);

% define the properties of the propagation medium
medium.sound_speed = 1500;	% [m/s]

% create initial pressure distribution using makeBall
ball_magnitude = 10;
ball_x_pos = 26;    % pixels
ball_y_pos = 22;    % pixels
ball_z_pos = 16;    % pixels
ball_radius = 3;    % pixels
p0_binary = ball_magnitude*makeBall(Nx, Ny, Nz, ball_x_pos, ball_y_pos, ball_z_pos, ball_radius);

% smooth the initial pressure distribution and restore the magnitude
p0 = smooth(kgrid, p0_binary, true);

% assign to the source structure
source.p0 = p0;

% define a Cartesian spherical sensor
sensor_radius = 4e-3;
num_sensor_points = 100;
center_pos = [0, 0, 0];
sensor_mask = makeCartSphere(sensor_radius, num_sensor_points, center_pos, true);

% assign to the sensor structure
sensor.mask = sensor_mask;

% create the time array
[kgrid.t_array dt] = makeTime(kgrid, medium.sound_speed);

% set the input arguements
input_args = {'PMLSize', PML_size, 'PMLInside', false, 'PlotPML', false, 'Smooth', false, 'DataCast', 'single'};

% run the simulation
sensor_data = kspaceFirstOrder3D(kgrid, medium, source, sensor, input_args{:});

% assign the reconstruction grid to be the same
kgrid_recon = kgrid;

% reset the initial pressure
source.p0 = 0;

% assign the time reversal data
sensor.time_reversal_boundary_data = sensor_data;

% run the time-reversal reconstruction
p0_recon = kspaceFirstOrder3D(kgrid, medium, source, sensor, input_args{:});

% create a binary sensor mask of an equivalent continuous sphere 
pixel_radius = round(sensor_radius/kgrid.dx);
binary_sensor_mask = makeSphere(kgrid_recon.Nx, kgrid_recon.Ny, kgrid_recon.Nz, pixel_radius);

% assign to the sensor structure
sensor.mask = binary_sensor_mask;

% interpolate data to remove the gaps and assign to time reversal data
sensor.time_reversal_boundary_data = interpCartData(kgrid_recon, sensor_data, sensor_mask, binary_sensor_mask);

% run the time-reversal reconstruction
p0_recon_interp = kspaceFirstOrder3D(kgrid, medium, source, sensor, input_args{:});

% =========================================================================
% VISUALISATION
% =========================================================================

% plot the initial pressure and sensor surface in voxel form
voxelPlot(double(p0_binary | cart2grid(kgrid, sensor_mask)));
view([60, 20]);

% plot the initial pressure
figure;
plot_scale = [-10 10];
subplot(2, 2, 1), imagesc(kgrid.x(1,:,1)*1e3, kgrid.z(:,1,1)*1e3, squeeze(p0(:, :, ball_y_pos)), plot_scale);
title('x-z plane');
axis image;
subplot(2, 2, 2), imagesc(kgrid.y(1,1,:)*1e3, kgrid.z(:,1,1)*1e3, squeeze(p0(:, ball_x_pos, :)), plot_scale);
title('y-z plane');
axis image;
xlabel('(All axes in mm)');
subplot(2, 2, 3), imagesc(kgrid.x(1,:,1)*1e3, kgrid.y(1,1,:)*1e3, squeeze(p0(ball_z_pos, :, :)).', plot_scale);
title('x-y plane');
axis image;
colormap(getColorMap);

% plot the reconstructed initial pressure
figure;
plot_scale = [-0.5, 0.5];
subplot(2, 2, 1), imagesc(kgrid.x(1,:,1)*1e3, kgrid.z(:,1,1)*1e3, squeeze(p0_recon(:, :, ball_y_pos)), plot_scale);
title('x-z plane');
axis image;
subplot(2, 2, 2), imagesc(kgrid.y(1,1,:)*1e3, kgrid.z(:,1,1)*1e3, squeeze(p0_recon(:, ball_x_pos, :)), plot_scale);
title('y-z plane');
axis image;
xlabel('(All axes in mm)');
subplot(2, 2, 3), imagesc(kgrid.x(1,:,1)*1e3, kgrid.y(1,1,:)*1e3, squeeze(p0_recon(ball_z_pos, :, :)).', plot_scale);
title('x-y plane');
axis image;
colormap(getColorMap);

% plot the reconstructed initial pressure
figure;
plot_scale = [-10 10];
subplot(2, 2, 1), imagesc(kgrid.x(1,:,1)*1e3, kgrid.z(:,1,1)*1e3, squeeze(p0_recon_interp(:, :, ball_y_pos)), plot_scale);
title('x-z plane');
axis image;
subplot(2, 2, 2), imagesc(kgrid.y(1,1,:)*1e3, kgrid.z(:,1,1)*1e3, squeeze(p0_recon_interp(:, ball_x_pos, :)), plot_scale);
title('y-z plane');
axis image;
xlabel('(All axes in mm)');
subplot(2, 2, 3), imagesc(kgrid.x(1,:,1)*1e3, kgrid.y(1,1,:)*1e3, squeeze(p0_recon_interp(ball_z_pos, :, :)).', plot_scale);
title('x-y plane');
axis image;
colormap(getColorMap);